<?php
// ============================================
// Simple EduMaster Admin Panel (PHP + MySQL)
// - Admin Login
// - Manage Classes, Subjects, Chapters
// - Manage Notes (PDF URL / Video URL)
// - Manage Quiz Questions
// ============================================

session_start();
require_once __DIR__ . '/config.php';

// ---- Simple admin login handling ----

if (isset($_POST['action']) && $_POST['action'] === 'login') {
    require_params(['email', 'password'], $_POST);
    $email = $_POST['email'];
    $password = $_POST['password'];

    $stmt = $pdo->prepare("SELECT id, name, email, password_hash FROM admins WHERE email = ?");
    $stmt->execute([$email]);
    $admin = $stmt->fetch();

    if ($admin && password_verify($password, $admin['password_hash'])) {
        $_SESSION['admin_id'] = $admin['id'];
        $_SESSION['admin_name'] = $admin['name'];
        header("Location: admin_panel.php");
        exit;
    } else {
        $login_error = "Invalid credentials";
    }
}

if (isset($_GET['logout'])) {
    session_destroy();
    header("Location: admin_panel.php");
    exit;
}

if (!isset($_SESSION['admin_id'])) {
    ?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>EduMaster Admin Login</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-dark text-light d-flex align-items-center justify-content-center" style="min-height:100vh;">
  <div class="card" style="min-width:380px;">
    <div class="card-body">
      <h3 class="mb-3 text-center">EduMaster Admin</h3>
      <?php if (isset($login_error)): ?>
        <div class="alert alert-danger"><?= htmlspecialchars($login_error) ?></div>
      <?php endif; ?>
      <form method="post">
        <input type="hidden" name="action" value="login">
        <div class="mb-3">
          <label class="form-label">Email</label>
          <input type="email" name="email" class="form-control" required>
        </div>
        <div class="mb-3">
          <label class="form-label">Password</label>
          <input type="password" name="password" class="form-control" required>
        </div>
        <button class="btn btn-primary w-100" type="submit">Login</button>
      </form>
    </div>
  </div>
</body>
</html>
<?php
    exit;
}

// ---- CRUD HANDLERS (very basic, no CSRF etc. for brevity) ----

// Handle bulk quiz upload (file upload)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['entity']) && $_POST['entity'] === 'bulk_quiz' && isset($_FILES['quiz_file'])) {
    try {
        $chapter_id = (int)$_POST['chapter_id'];
        $file = $_FILES['quiz_file'];
        $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        
        if ($ext === 'txt') {
            $content = file_get_contents($file['tmp_name']);
            $lines = explode("\n", $content);
            $count = 0;
            foreach ($lines as $line) {
                $line = trim($line);
                if (empty($line)) continue;
                $parts = explode('|', $line);
                if (count($parts) >= 6) {
                    $stmt = $pdo->prepare("INSERT INTO quiz_questions (chapter_id, question_text, option_a, option_b, option_c, option_d, correct_option, explanation) VALUES (?,?,?,?,?,?,?,?)");
                    $stmt->execute([
                        $chapter_id,
                        $parts[0],
                        $parts[1] ?? '',
                        $parts[2] ?? '',
                        $parts[3] ?? '',
                        $parts[4] ?? '',
                        strtoupper($parts[5] ?? 'A'),
                        $parts[6] ?? ''
                    ]);
                    $count++;
                }
            }
            $crud_success = "Uploaded $count questions successfully!";
        } elseif ($ext === 'csv') {
            $handle = fopen($file['tmp_name'], 'r');
            $count = 0;
            while (($data = fgetcsv($handle)) !== FALSE) {
                if (count($data) >= 6) {
                    $stmt = $pdo->prepare("INSERT INTO quiz_questions (chapter_id, question_text, option_a, option_b, option_c, option_d, correct_option, explanation) VALUES (?,?,?,?,?,?,?,?)");
                    $stmt->execute([
                        $chapter_id,
                        $data[0],
                        $data[1] ?? '',
                        $data[2] ?? '',
                        $data[3] ?? '',
                        $data[4] ?? '',
                        strtoupper($data[5] ?? 'A'),
                        $data[6] ?? ''
                    ]);
                    $count++;
                }
            }
            fclose($handle);
            $crud_success = "Uploaded $count questions from CSV!";
        } else {
            $crud_error = "Please upload .txt or .csv file only.";
        }
    } catch (PDOException $e) {
        $crud_error = $e->getMessage();
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['entity'])) {
    $entity = $_POST['entity'];

    try {
        if ($entity === 'class') {
            $name = $_POST['name'];
            $description = $_POST['description'] ?? '';
            $status = $_POST['status'] ?? 'active';
            if (!empty($_POST['id'])) {
                $stmt = $pdo->prepare("UPDATE classes SET name=?, description=?, status=? WHERE id=?");
                $stmt->execute([$name, $description, $status, (int)$_POST['id']]);
            } else {
                $stmt = $pdo->prepare("INSERT INTO classes (name, description, status) VALUES (?, ?, ?)");
                $stmt->execute([$name, $description, $status]);
            }
        }

        if ($entity === 'subject') {
            $class_id = (int)$_POST['class_id'];
            $name = $_POST['name'];
            $description = $_POST['description'] ?? '';
            if (!empty($_POST['id'])) {
                $stmt = $pdo->prepare("UPDATE subjects SET class_id=?, name=?, description=? WHERE id=?");
                $stmt->execute([$class_id, $name, $description, (int)$_POST['id']]);
            } else {
                $stmt = $pdo->prepare("INSERT INTO subjects (class_id, name, description) VALUES (?, ?, ?)");
                $stmt->execute([$class_id, $name, $description]);
            }
        }

        if ($entity === 'chapter') {
            $subject_id = (int)$_POST['subject_id'];
            $title = $_POST['title'];
            $description = $_POST['description'] ?? '';
            if (!empty($_POST['id'])) {
                $stmt = $pdo->prepare("UPDATE chapters SET subject_id=?, title=?, description=? WHERE id=?");
                $stmt->execute([$subject_id, $title, $description, (int)$_POST['id']]);
            } else {
                $stmt = $pdo->prepare("INSERT INTO chapters (subject_id, title, description) VALUES (?, ?, ?)");
                $stmt->execute([$subject_id, $title, $description]);
            }
        }

        if ($entity === 'note') {
            $chapter_id = (int)$_POST['chapter_id'];
            $pdf_url = $_POST['pdf_url'] ?? '';
            $video_url = $_POST['video_url'] ?? '';
            $text_content = $_POST['text_content'] ?? '';
            if (!empty($_POST['id'])) {
                $stmt = $pdo->prepare("UPDATE notes SET chapter_id=?, pdf_url=?, video_url=?, text_content=? WHERE id=?");
                $stmt->execute([$chapter_id, $pdf_url, $video_url, $text_content, (int)$_POST['id']]);
            } else {
                $stmt = $pdo->prepare("INSERT INTO notes (chapter_id, pdf_url, video_url, text_content) VALUES (?, ?, ?, ?)");
                $stmt->execute([$chapter_id, $pdf_url, $video_url, $text_content]);
            }
        }
        
        // Bulk quiz upload
        if ($entity === 'bulk_quiz' && isset($_FILES['quiz_file'])) {
            $chapter_id = (int)$_POST['chapter_id'];
            $file = $_FILES['quiz_file'];
            $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
            
            if ($ext === 'txt') {
                // Text file format: Question|A|B|C|D|Correct|Explanation
                $content = file_get_contents($file['tmp_name']);
                $lines = explode("\n", $content);
                $count = 0;
                foreach ($lines as $line) {
                    $line = trim($line);
                    if (empty($line)) continue;
                    $parts = explode('|', $line);
                    if (count($parts) >= 6) {
                        $stmt = $pdo->prepare("INSERT INTO quiz_questions (chapter_id, question_text, option_a, option_b, option_c, option_d, correct_option, explanation) VALUES (?,?,?,?,?,?,?,?)");
                        $stmt->execute([
                            $chapter_id,
                            $parts[0],
                            $parts[1] ?? '',
                            $parts[2] ?? '',
                            $parts[3] ?? '',
                            $parts[4] ?? '',
                            strtoupper($parts[5] ?? 'A'),
                            $parts[6] ?? ''
                        ]);
                        $count++;
                    }
                }
                $crud_success = "Uploaded $count questions successfully!";
            } elseif ($ext === 'csv' || $ext === 'xlsx' || $ext === 'xls') {
                // Excel/CSV support
                if ($ext === 'csv') {
                    $handle = fopen($file['tmp_name'], 'r');
                    $count = 0;
                    while (($data = fgetcsv($handle)) !== FALSE) {
                        if (count($data) >= 6) {
                            $stmt = $pdo->prepare("INSERT INTO quiz_questions (chapter_id, question_text, option_a, option_b, option_c, option_d, correct_option, explanation) VALUES (?,?,?,?,?,?,?,?)");
                            $stmt->execute([
                                $chapter_id,
                                $data[0],
                                $data[1] ?? '',
                                $data[2] ?? '',
                                $data[3] ?? '',
                                $data[4] ?? '',
                                strtoupper($data[5] ?? 'A'),
                                $data[6] ?? ''
                            ]);
                            $count++;
                        }
                    }
                    fclose($handle);
                    $crud_success = "Uploaded $count questions from CSV!";
                } else {
                    $crud_error = "Excel files (.xlsx/.xls) require PHPExcel library. Please use CSV or TXT format.";
                }
            }
        }
        
        // Ads management
        if ($entity === 'ad') {
            $ad_type = $_POST['ad_type'];
            $ad_unit_id = $_POST['ad_unit_id'];
            $status = $_POST['status'] ?? 'active';
            $position = $_POST['position'] ?? '';
            if (!empty($_POST['id'])) {
                $stmt = $pdo->prepare("UPDATE ads_settings SET ad_type=?, ad_unit_id=?, status=?, position=? WHERE id=?");
                $stmt->execute([$ad_type, $ad_unit_id, $status, $position, (int)$_POST['id']]);
            } else {
                $stmt = $pdo->prepare("INSERT INTO ads_settings (ad_type, ad_unit_id, status, position) VALUES (?,?,?,?)");
                $stmt->execute([$ad_type, $ad_unit_id, $status, $position]);
            }
        }

        if ($entity === 'question') {
            $chapter_id = (int)$_POST['chapter_id'];
            $question_text = $_POST['question_text'];
            $option_a = $_POST['option_a'];
            $option_b = $_POST['option_b'];
            $option_c = $_POST['option_c'];
            $option_d = $_POST['option_d'];
            $correct_option = $_POST['correct_option'];
            $explanation = $_POST['explanation'] ?? '';
            if (!empty($_POST['id'])) {
                $stmt = $pdo->prepare("UPDATE quiz_questions SET chapter_id=?, question_text=?, option_a=?, option_b=?, option_c=?, option_d=?, correct_option=?, explanation=? WHERE id=?");
                $stmt->execute([$chapter_id, $question_text, $option_a, $option_b, $option_c, $option_d, $correct_option, $explanation, (int)$_POST['id']]);
            } else {
                $stmt = $pdo->prepare("INSERT INTO quiz_questions (chapter_id, question_text, option_a, option_b, option_c, option_d, correct_option, explanation) VALUES (?,?,?,?,?,?,?,?)");
                $stmt->execute([$chapter_id, $question_text, $option_a, $option_b, $option_c, $option_d, $correct_option, $explanation]);
            }
        }
    } catch (PDOException $e) {
        $crud_error = $e->getMessage();
    }
}

// ---- Fetch lists for forms/tables ----

$classes = $pdo->query("SELECT * FROM classes ORDER BY id ASC")->fetchAll();
$subjects = $pdo->query("SELECT * FROM subjects ORDER BY id ASC")->fetchAll();
$chapters = $pdo->query("SELECT * FROM chapters ORDER BY id ASC")->fetchAll();

// Helper: map id => row
function index_by_id($rows)
{
    $out = [];
    foreach ($rows as $r) {
        $out[$r['id']] = $r;
    }
    return $out;
}

$classesById = index_by_id($classes);
$subjectsById = index_by_id($subjects);

?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>EduMaster Admin Panel (PHP/MySQL)</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light">
  <nav class="navbar navbar-expand-lg navbar-dark bg-dark mb-3">
    <div class="container-fluid">
      <a class="navbar-brand" href="#">EduMaster Admin</a>
      <div class="d-flex">
        <span class="navbar-text me-3">Welcome, <?= htmlspecialchars($_SESSION['admin_name']) ?></span>
        <a class="btn btn-outline-light btn-sm" href="?logout=1">Logout</a>
      </div>
    </div>
  </nav>

  <div class="container mb-5">
    <?php if (isset($crud_error)): ?>
      <div class="alert alert-danger"><?= htmlspecialchars($crud_error) ?></div>
    <?php endif; ?>
    <?php if (isset($crud_success)): ?>
      <div class="alert alert-success"><?= htmlspecialchars($crud_success) ?></div>
    <?php endif; ?>
    
    <!-- API Connection Info -->
    <div class="card mb-3 border-info">
      <div class="card-header bg-info text-white">
        <strong>📡 API Connection Details (for Android App)</strong>
      </div>
      <div class="card-body">
        <?php
          $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
          $host = $_SERVER['HTTP_HOST'];
          $base_path = dirname($_SERVER['PHP_SELF']);
          $api_url = $protocol . '://' . $host . $base_path . '/api.php';
          $base_url = $protocol . '://' . $host . $base_path . '/';
          
          // Get admin details
          $admin_stmt = $pdo->prepare("SELECT id, name, email FROM admins WHERE id = ?");
          $admin_stmt->execute([$_SESSION['admin_id']]);
          $current_admin = $admin_stmt->fetch();
        ?>
        <div class="row">
          <div class="col-md-6">
            <h6 class="text-primary">API Base URL:</h6>
            <div class="input-group mb-2">
              <input type="text" class="form-control" id="apiBaseUrl" value="<?= htmlspecialchars($base_url) ?>" readonly>
              <button class="btn btn-outline-secondary" type="button" onclick="copyToClipboard('apiBaseUrl')">Copy</button>
            </div>
            <small class="text-muted">Use this in Android: ApiClient.java → BASE_URL</small>
          </div>
          <div class="col-md-6">
            <h6 class="text-primary">API Endpoint:</h6>
            <div class="input-group mb-2">
              <input type="text" class="form-control" id="apiEndpoint" value="<?= htmlspecialchars($api_url) ?>" readonly>
              <button class="btn btn-outline-secondary" type="button" onclick="copyToClipboard('apiEndpoint')">Copy</button>
            </div>
            <small class="text-muted">Test: <a href="<?= htmlspecialchars($api_url) ?>?action=classes" target="_blank">Click here</a></small>
          </div>
        </div>
        <hr>
        <div class="row">
          <div class="col-md-12">
            <h6 class="text-success">Your Admin Details:</h6>
            <table class="table table-sm table-bordered">
              <tr>
                <th width="150">Admin ID:</th>
                <td><strong><?= htmlspecialchars($current_admin['id'] ?? 'N/A') ?></strong></td>
              </tr>
              <tr>
                <th>Name:</th>
                <td><?= htmlspecialchars($current_admin['name'] ?? 'N/A') ?></td>
              </tr>
              <tr>
                <th>Email (Username):</th>
                <td><?= htmlspecialchars($current_admin['email'] ?? 'N/A') ?></td>
              </tr>
            </table>
          </div>
        </div>
        <div class="alert alert-info mt-3">
          <strong>💡 Tip:</strong> Copy the API Base URL and paste it in Android Studio → <code>ApiClient.java</code> → Replace <code>BASE_URL</code> constant
        </div>
      </div>
    </div>

    <!-- Classes -->
    <div class="card mb-3">
      <div class="card-header d-flex justify-content-between align-items-center">
        <span>Classes</span>
      </div>
      <div class="card-body">
        <form class="row g-2 mb-2" method="post">
          <input type="hidden" name="entity" value="class">
          <div class="col-md-3">
            <input type="text" name="name" class="form-control" placeholder="Class name" required>
          </div>
          <div class="col-md-4">
            <input type="text" name="description" class="form-control" placeholder="Description">
          </div>
          <div class="col-md-3">
            <select name="status" class="form-select">
              <option value="active">Active</option>
              <option value="inactive">Inactive</option>
            </select>
          </div>
          <div class="col-md-2">
            <button class="btn btn-primary w-100" type="submit">Add</button>
          </div>
        </form>
        <table class="table table-sm">
          <thead><tr><th>ID</th><th>Name</th><th>Status</th></tr></thead>
          <tbody>
            <?php foreach ($classes as $c): ?>
              <tr>
                <td><?= $c['id'] ?></td>
                <td><?= htmlspecialchars($c['name']) ?></td>
                <td><?= htmlspecialchars($c['status']) ?></td>
              </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      </div>
    </div>

    <!-- Subjects -->
    <div class="card mb-3">
      <div class="card-header">Subjects</div>
      <div class="card-body">
        <form class="row g-2 mb-2" method="post">
          <input type="hidden" name="entity" value="subject">
          <div class="col-md-3">
            <select name="class_id" class="form-select" required>
              <option value="">Class</option>
              <?php foreach ($classes as $c): ?>
                <option value="<?= $c['id'] ?>"><?= htmlspecialchars($c['name']) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-3">
            <input type="text" name="name" class="form-control" placeholder="Subject name" required>
          </div>
          <div class="col-md-4">
            <input type="text" name="description" class="form-control" placeholder="Description">
          </div>
          <div class="col-md-2">
            <button class="btn btn-primary w-100" type="submit">Add</button>
          </div>
        </form>
        <table class="table table-sm">
          <thead><tr><th>ID</th><th>Class</th><th>Name</th></tr></thead>
          <tbody>
            <?php foreach ($subjects as $s): ?>
              <tr>
                <td><?= $s['id'] ?></td>
                <td><?= isset($classesById[$s['class_id']]) ? htmlspecialchars($classesById[$s['class_id']]['name']) : $s['class_id'] ?></td>
                <td><?= htmlspecialchars($s['name']) ?></td>
              </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      </div>
    </div>

    <!-- Chapters -->
    <div class="card mb-3">
      <div class="card-header">Chapters</div>
      <div class="card-body">
        <form class="row g-2 mb-2" method="post">
          <input type="hidden" name="entity" value="chapter">
          <div class="col-md-3">
            <select name="subject_id" class="form-select" required>
              <option value="">Subject</option>
              <?php foreach ($subjects as $s): ?>
                <option value="<?= $s['id'] ?>"><?= htmlspecialchars($s['name']) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-3">
            <input type="text" name="title" class="form-control" placeholder="Chapter title" required>
          </div>
          <div class="col-md-4">
            <input type="text" name="description" class="form-control" placeholder="Description">
          </div>
          <div class="col-md-2">
            <button class="btn btn-primary w-100" type="submit">Add</button>
          </div>
        </form>
        <table class="table table-sm">
          <thead><tr><th>ID</th><th>Subject</th><th>Title</th></tr></thead>
          <tbody>
            <?php foreach ($chapters as $ch): ?>
              <tr>
                <td><?= $ch['id'] ?></td>
                <td><?= isset($subjectsById[$ch['subject_id']]) ? htmlspecialchars($subjectsById[$ch['subject_id']]['name']) : $ch['subject_id'] ?></td>
                <td><?= htmlspecialchars($ch['title']) ?></td>
              </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      </div>
    </div>

    <!-- Notes -->
    <div class="card mb-3">
      <div class="card-header">Notes (PDF / Video / Text)</div>
      <div class="card-body">
        <form method="post">
          <input type="hidden" name="entity" value="note">
          <div class="row g-2 mb-2">
            <div class="col-md-3">
              <select name="chapter_id" class="form-select" required>
                <option value="">Chapter</option>
                <?php foreach ($chapters as $ch): ?>
                  <option value="<?= $ch['id'] ?>"><?= htmlspecialchars($ch['title']) ?></option>
                <?php endforeach; ?>
              </select>
            </div>
            <div class="col-md-3">
              <input type="text" name="pdf_url" class="form-control" placeholder="PDF URL">
            </div>
            <div class="col-md-3">
              <input type="text" name="video_url" class="form-control" placeholder="YouTube URL">
            </div>
            <div class="col-md-3">
              <button class="btn btn-primary w-100" type="submit">Add</button>
            </div>
          </div>
          <div class="mb-2">
            <label class="form-label">Text Content (for reading notes):</label>
            <textarea name="text_content" class="form-control" rows="5" placeholder="Enter text content here..."></textarea>
          </div>
        </form>
        <?php
          $notes = $pdo->query("SELECT * FROM notes ORDER BY id DESC LIMIT 50")->fetchAll();
        ?>
        <table class="table table-sm">
          <thead><tr><th>ID</th><th>Chapter</th><th>PDF</th><th>Video</th><th>Text</th></tr></thead>
          <tbody>
            <?php foreach ($notes as $n): ?>
              <tr>
                <td><?= $n['id'] ?></td>
                <td><?= $n['chapter_id'] ?></td>
                <td><?= $n['pdf_url'] ? '<a href="'.htmlspecialchars($n['pdf_url']).'" target="_blank">PDF</a>' : '-' ?></td>
                <td><?= $n['video_url'] ? '<a href="'.htmlspecialchars($n['video_url']).'" target="_blank">Video</a>' : '-' ?></td>
                <td><?= isset($n['text_content']) && $n['text_content'] ? 'Yes ('.strlen($n['text_content']).' chars)' : '-' ?></td>
              </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      </div>
    </div>

    <!-- Quiz Questions -->
    <div class="card mb-3">
      <div class="card-header">Quiz Questions</div>
      <div class="card-body">
        <form class="row g-2 mb-2" method="post">
          <input type="hidden" name="entity" value="question">
          <div class="col-md-3">
            <select name="chapter_id" class="form-select" required>
              <option value="">Chapter</option>
              <?php foreach ($chapters as $ch): ?>
                <option value="<?= $ch['id'] ?>"><?= htmlspecialchars($ch['title']) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-9">
            <input type="text" name="question_text" class="form-control" placeholder="Question text" required>
          </div>
          <div class="col-md-3">
            <input type="text" name="option_a" class="form-control" placeholder="Option A" required>
          </div>
          <div class="col-md-3">
            <input type="text" name="option_b" class="form-control" placeholder="Option B" required>
          </div>
          <div class="col-md-3">
            <input type="text" name="option_c" class="form-control" placeholder="Option C" required>
          </div>
          <div class="col-md-3">
            <input type="text" name="option_d" class="form-control" placeholder="Option D" required>
          </div>
          <div class="col-md-3">
            <select name="correct_option" class="form-select">
              <option value="A">A</option>
              <option value="B">B</option>
              <option value="C">C</option>
              <option value="D">D</option>
            </select>
          </div>
          <div class="col-md-7">
            <input type="text" name="explanation" class="form-control" placeholder="Explanation (optional)">
          </div>
          <div class="col-md-2">
            <button class="btn btn-primary w-100" type="submit">Add</button>
          </div>
        </form>
        <?php
          $questions = $pdo->query("SELECT * FROM quiz_questions ORDER BY id DESC LIMIT 50")->fetchAll();
        ?>
        <table class="table table-sm">
          <thead><tr><th>ID</th><th>Chapter</th><th>Question</th><th>Correct</th></tr></thead>
          <tbody>
            <?php foreach ($questions as $q): ?>
              <tr>
                <td><?= $q['id'] ?></td>
                <td><?= $q['chapter_id'] ?></td>
                <td><?= htmlspecialchars($q['question_text']) ?></td>
                <td><?= htmlspecialchars($q['correct_option']) ?></td>
              </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      </div>
    </div>

    <!-- Bulk Quiz Upload -->
    <div class="card mb-3">
      <div class="card-header">
        Bulk Quiz Upload
        <a href="quiz_format_template.txt" download class="btn btn-sm btn-outline-primary float-end">Download Format Template</a>
      </div>
      <div class="card-body">
        <div class="alert alert-info">
          <strong>Format:</strong> Text file (.txt) with pipe-separated values<br>
          Format: Question|Option A|Option B|Option C|Option D|Correct Answer (A/B/C/D)|Explanation<br>
          <strong>OR</strong> CSV file with same columns
        </div>
        <form method="post" enctype="multipart/form-data">
          <input type="hidden" name="entity" value="bulk_quiz">
          <div class="row g-2">
            <div class="col-md-3">
              <select name="chapter_id" class="form-select" required>
                <option value="">Select Chapter</option>
                <?php foreach ($chapters as $ch): ?>
                  <option value="<?= $ch['id'] ?>"><?= htmlspecialchars($ch['title']) ?></option>
                <?php endforeach; ?>
              </select>
            </div>
            <div class="col-md-6">
              <input type="file" name="quiz_file" class="form-control" accept=".txt,.csv" required>
            </div>
            <div class="col-md-3">
              <button class="btn btn-success w-100" type="submit">Upload Questions</button>
            </div>
          </div>
        </form>
      </div>
    </div>

    <!-- Ads Management -->
    <div class="card mb-3">
      <div class="card-header">AdMob Ads Management</div>
      <div class="card-body">
        <form method="post">
          <input type="hidden" name="entity" value="ad">
          <div class="row g-2 mb-2">
            <div class="col-md-3">
              <select name="ad_type" class="form-select" required>
                <option value="banner">Banner Ad</option>
                <option value="interstitial">Interstitial Ad</option>
                <option value="rewarded">Rewarded Ad</option>
              </select>
            </div>
            <div class="col-md-4">
              <input type="text" name="ad_unit_id" class="form-control" placeholder="Ad Unit ID (ca-app-pub-...)" required>
            </div>
            <div class="col-md-2">
              <select name="status" class="form-select">
                <option value="active">Active</option>
                <option value="inactive">Inactive</option>
              </select>
            </div>
            <div class="col-md-2">
              <input type="text" name="position" class="form-control" placeholder="Position (e.g., home_bottom)">
            </div>
            <div class="col-md-1">
              <button class="btn btn-primary w-100" type="submit">Add</button>
            </div>
          </div>
        </form>
        <?php
          try {
            $ads = $pdo->query("SELECT * FROM ads_settings ORDER BY id DESC")->fetchAll();
          } catch (PDOException $e) {
            $ads = [];
          }
        ?>
        <table class="table table-sm">
          <thead><tr><th>ID</th><th>Type</th><th>Ad Unit ID</th><th>Status</th><th>Position</th></tr></thead>
          <tbody>
            <?php if (empty($ads)): ?>
              <tr><td colspan="5" class="text-center text-muted">No ads configured yet</td></tr>
            <?php else: ?>
              <?php foreach ($ads as $ad): ?>
                <tr>
                  <td><?= $ad['id'] ?></td>
                  <td><?= htmlspecialchars($ad['ad_type']) ?></td>
                  <td><small><?= htmlspecialchars($ad['ad_unit_id']) ?></small></td>
                  <td><?= htmlspecialchars($ad['status']) ?></td>
                  <td><?= htmlspecialchars($ad['position']) ?></td>
                </tr>
              <?php endforeach; ?>
            <?php endif; ?>
          </tbody>
        </table>
      </div>
    </div>
  </div>
  
  <script>
  function copyToClipboard(elementId) {
    var copyText = document.getElementById(elementId);
    copyText.select();
    copyText.setSelectionRange(0, 99999);
    document.execCommand("copy");
    alert("Copied to clipboard!");
  }
  </script>
</body>
</html>

