<?php
// ============================================
// EduMaster REST API (PHP + MySQL)
// ============================================

require_once __DIR__ . '/config.php';

$action = $_GET['action'] ?? '';

switch ($action) {
    // --------------------------
    // Auth endpoints
    // --------------------------
    case 'student_register':
        student_register($pdo);
        break;
    case 'student_login':
        student_login($pdo);
        break;

    // --------------------------
    // Data fetch for Android
    // --------------------------
    case 'classes':
        get_classes($pdo);
        break;
    case 'subjects':
        get_subjects($pdo);
        break;
    case 'chapters':
        get_chapters($pdo);
        break;
    case 'notes':
        get_notes($pdo);
        break;
    case 'quiz_questions':
        get_quiz_questions($pdo);
        break;
    case 'submit_quiz':
        submit_quiz($pdo);
        break;
    case 'leaderboard':
        get_leaderboard($pdo);
        break;
    case 'ads_settings':
        get_ads_settings($pdo);
        break;

    default:
        json_response(['error' => 'Invalid action'], 400);
}

function get_ads_settings(PDO $pdo)
{
    try {
        $stmt = $pdo->query("SELECT ad_type, ad_unit_id, status, position FROM ads_settings WHERE status = 'active'");
        json_response(['ads' => $stmt->fetchAll()]);
    } catch (PDOException $e) {
        json_response(['ads' => []]);
    }
}

// ============================================
// Endpoint Implementations
// ============================================

function student_register(PDO $pdo)
{
    $input = json_decode(file_get_contents('php://input'), true) ?? $_POST;
    require_params(['name', 'email', 'password', 'class_id'], $input);

    $name = $input['name'];
    $email = $input['email'];
    $password = $input['password'];
    $class_id = (int)$input['class_id'];

    $hash = password_hash($password, PASSWORD_BCRYPT);

    try {
        $stmt = $pdo->prepare("INSERT INTO students (name, email, password_hash, class_id) VALUES (?, ?, ?, ?)");
        $stmt->execute([$name, $email, $hash, $class_id]);
        json_response(['success' => true, 'student_id' => $pdo->lastInsertId()]);
    } catch (PDOException $e) {
        if ($e->getCode() === '23000') {
            json_response(['error' => 'Email already registered'], 409);
        }
        json_response(['error' => 'Registration failed', 'details' => $e->getMessage()], 500);
    }
}

function student_login(PDO $pdo)
{
    $input = json_decode(file_get_contents('php://input'), true) ?? $_POST;
    require_params(['email', 'password'], $input);

    $email = $input['email'];
    $password = $input['password'];

    $stmt = $pdo->prepare("SELECT id, name, email, password_hash, class_id FROM students WHERE email = ?");
    $stmt->execute([$email]);
    $student = $stmt->fetch();

    if (!$student || !password_verify($password, $student['password_hash'])) {
        json_response(['error' => 'Invalid email or password'], 401);
    }

    unset($student['password_hash']);
    json_response(['success' => true, 'student' => $student]);
}

function get_classes(PDO $pdo)
{
    $stmt = $pdo->query("SELECT id, name, description, status FROM classes WHERE status = 'active' ORDER BY id ASC");
    $classes = $stmt->fetchAll();
    json_response(['classes' => $classes]);
}

function get_subjects(PDO $pdo)
{
    $class_id = isset($_GET['class_id']) ? (int)$_GET['class_id'] : 0;
    if (!$class_id) {
        json_response(['error' => 'class_id required'], 400);
    }
    $stmt = $pdo->prepare("SELECT id, class_id, name, description FROM subjects WHERE class_id = ? ORDER BY id ASC");
    $stmt->execute([$class_id]);
    json_response(['subjects' => $stmt->fetchAll()]);
}

function get_chapters(PDO $pdo)
{
    $subject_id = isset($_GET['subject_id']) ? (int)$_GET['subject_id'] : 0;
    if (!$subject_id) {
        json_response(['error' => 'subject_id required'], 400);
    }
    $stmt = $pdo->prepare("SELECT id, subject_id, title, description FROM chapters WHERE subject_id = ? ORDER BY id ASC");
    $stmt->execute([$subject_id]);
    json_response(['chapters' => $stmt->fetchAll()]);
}

function get_notes(PDO $pdo)
{
    $chapter_id = isset($_GET['chapter_id']) ? (int)$_GET['chapter_id'] : 0;
    if (!$chapter_id) {
        json_response(['error' => 'chapter_id required'], 400);
    }
    $stmt = $pdo->prepare("SELECT id, chapter_id, pdf_url, video_url, text_content, created_at FROM notes WHERE chapter_id = ?");
    $stmt->execute([$chapter_id]);
    json_response(['notes' => $stmt->fetchAll()]);
}

function get_ads_settings(PDO $pdo)
{
    $stmt = $pdo->query("SELECT ad_type, ad_unit_id, status, position FROM ads_settings WHERE status = 'active'");
    json_response(['ads' => $stmt->fetchAll()]);
}

function get_quiz_questions(PDO $pdo)
{
    $chapter_id = isset($_GET['chapter_id']) ? (int)$_GET['chapter_id'] : 0;
    if (!$chapter_id) {
        json_response(['error' => 'chapter_id required'], 400);
    }
    $stmt = $pdo->prepare("SELECT id, chapter_id, question_text, option_a, option_b, option_c, option_d, correct_option, explanation FROM quiz_questions WHERE chapter_id = ?");
    $stmt->execute([$chapter_id]);
    json_response(['questions' => $stmt->fetchAll()]);
}

function submit_quiz(PDO $pdo)
{
    $input = json_decode(file_get_contents('php://input'), true) ?? $_POST;
    require_params(['student_id', 'chapter_id', 'score', 'total'], $input);

    $student_id = (int)$input['student_id'];
    $chapter_id = (int)$input['chapter_id'];
    $score = (int)$input['score'];
    $total = (int)$input['total'];

    $stmt = $pdo->prepare("INSERT INTO quiz_results (student_id, chapter_id, score, total) VALUES (?, ?, ?, ?)");
    $stmt->execute([$student_id, $chapter_id, $score, $total]);

    json_response(['success' => true]);
}

function get_leaderboard(PDO $pdo)
{
    // Average score per student across all attempts
    $sql = "SELECT s.id, s.name, s.class_id, AVG(r.score) AS avg_score
            FROM students s
            JOIN quiz_results r ON r.student_id = s.id
            GROUP BY s.id, s.name, s.class_id
            ORDER BY avg_score DESC
            LIMIT 50";
    $stmt = $pdo->query($sql);
    json_response(['leaderboard' => $stmt->fetchAll()]);
}

