<?php
/**
 * EduMaster Installation Script
 * Run this file once to set up the database and configuration
 */

// Check if already installed
if (file_exists('config.php') && file_exists('INSTALLED')) {
    die('Already installed! Delete INSTALLED file to reinstall.');
}

$step = isset($_GET['step']) ? (int)$_GET['step'] : 1;
$error = '';
$success = '';

// Step 2: Database Setup
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['install'])) {
    $db_host = $_POST['db_host'] ?? 'localhost';
    $db_name = $_POST['db_name'] ?? 'edumaster_db';
    $db_user = $_POST['db_user'] ?? 'root';
    $db_pass = $_POST['db_pass'] ?? '';
    
    // Super Admin Details
    $admin_name = $_POST['admin_name'] ?? 'Super Admin';
    $admin_email = $_POST['admin_email'] ?? 'admin@edumaster.com';
    $admin_password = $_POST['admin_password'] ?? 'admin123';
    $admin_id = $_POST['admin_id'] ?? '1';
    
    try {
        // Test connection
        $pdo = new PDO("mysql:host=$db_host;charset=utf8mb4", $db_user, $db_pass, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION
        ]);
        
        // Create database if not exists
        $pdo->exec("CREATE DATABASE IF NOT EXISTS `$db_name` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
        $pdo->exec("USE `$db_name`");
        
        // Read and execute schema
        $schema = file_get_contents('schema.sql');
        $statements = array_filter(array_map('trim', explode(';', $schema)));
        
        foreach ($statements as $statement) {
            if (!empty($statement)) {
                $pdo->exec($statement);
            }
        }
        
        // Create super admin user
        $admin_hash = password_hash($admin_password, PASSWORD_BCRYPT);
        $stmt = $pdo->prepare("INSERT INTO admins (id, name, email, password_hash) VALUES (?, ?, ?, ?)");
        $stmt->execute([(int)$admin_id, $admin_name, $admin_email, $admin_hash]);
        
        // Create config.php
        $config_content = "<?php
// ============================================
// Database configuration for EduMaster API
// ============================================

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if (\$_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

\$DB_HOST = '$db_host';
\$DB_NAME = '$db_name';
\$DB_USER = '$db_user';
\$DB_PASS = '$db_pass';

try {
    \$pdo = new PDO(\"mysql:host=\$DB_HOST;dbname=\$DB_NAME;charset=utf8mb4\", \$DB_USER, \$DB_PASS, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);
} catch (PDOException \$e) {
    http_response_code(500);
    echo json_encode(['error' => 'Database connection failed', 'details' => \$e->getMessage()]);
    exit;
}

function json_response(\$data, \$code = 200)
{
    http_response_code(\$code);
    echo json_encode(\$data);
    exit;
}

function require_params(\$params, \$source)
{
    foreach (\$params as \$p) {
        if (!isset(\$source[\$p]) || \$source[\$p] === '') {
            json_response(['error' => \"Missing parameter: \$p\"], 400);
        }
    }
}
";
        file_put_contents('config.php', $config_content);
        
        // Create INSTALLED marker
        file_put_contents('INSTALLED', date('Y-m-d H:i:s'));
        
        $success = 'Installation completed successfully!';
        $step = 3;
    } catch (Exception $e) {
        $error = 'Installation failed: ' . $e->getMessage();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>EduMaster Installation</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        .container {
            background: white;
            border-radius: 12px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            max-width: 600px;
            width: 100%;
            padding: 40px;
        }
        h1 {
            color: #667eea;
            margin-bottom: 10px;
            font-size: 28px;
        }
        .subtitle {
            color: #666;
            margin-bottom: 30px;
        }
        .form-group {
            margin-bottom: 20px;
        }
        label {
            display: block;
            margin-bottom: 8px;
            color: #333;
            font-weight: 500;
        }
        input {
            width: 100%;
            padding: 12px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            font-size: 14px;
            transition: border-color 0.3s;
        }
        input:focus {
            outline: none;
            border-color: #667eea;
        }
        .btn {
            background: #667eea;
            color: white;
            padding: 14px 30px;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            width: 100%;
            transition: background 0.3s;
        }
        .btn:hover {
            background: #5568d3;
        }
        .error {
            background: #fee;
            color: #c33;
            padding: 12px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        .success {
            background: #efe;
            color: #3c3;
            padding: 12px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        .info {
            background: #e3f2fd;
            color: #1976d2;
            padding: 12px;
            border-radius: 8px;
            margin-bottom: 20px;
            font-size: 14px;
        }
        .step-indicator {
            display: flex;
            justify-content: space-between;
            margin-bottom: 30px;
        }
        .step {
            flex: 1;
            text-align: center;
            padding: 10px;
            background: #f0f0f0;
            margin: 0 5px;
            border-radius: 8px;
            font-size: 12px;
        }
        .step.active {
            background: #667eea;
            color: white;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🚀 EduMaster Installation</h1>
        <p class="subtitle">Complete setup in 2 minutes</p>
        
        <div class="step-indicator">
            <div class="step <?= $step >= 1 ? 'active' : '' ?>">1. Welcome</div>
            <div class="step <?= $step >= 2 ? 'active' : '' ?>">2. Database</div>
            <div class="step <?= $step >= 3 ? 'active' : '' ?>">3. Complete</div>
        </div>
        
        <?php if ($error): ?>
            <div class="error"><?= htmlspecialchars($error) ?></div>
        <?php endif; ?>
        
        <?php if ($success): ?>
            <div class="success"><?= htmlspecialchars($success) ?></div>
            <div class="info">
                <strong>Installation Complete!</strong><br><br>
                <strong>Admin Panel:</strong> <a href="admin_panel.php" target="_blank">admin_panel.php</a><br>
                <strong>API Endpoint:</strong> <a href="api.php" target="_blank">api.php</a><br><br>
                <strong>Super Admin Credentials:</strong><br>
                <strong>ID:</strong> <?= htmlspecialchars($admin_id ?? '1') ?><br>
                <strong>Name:</strong> <?= htmlspecialchars($admin_name ?? 'Super Admin') ?><br>
                <strong>Email:</strong> <?= htmlspecialchars($admin_email ?? 'admin@edumaster.com') ?><br>
                <strong>Password:</strong> <?= htmlspecialchars($admin_password ?? 'admin123') ?><br><br>
                <strong>API Base URL for Android App:</strong><br>
                <div style="background: #f5f5f5; padding: 10px; border-radius: 5px; margin: 10px 0;">
                    <code style="font-size: 14px;"><?= (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'] . dirname($_SERVER['PHP_SELF']) . '/' ?></code>
                </div>
                <small>Copy this and paste in Android: <code>ApiClient.java</code> → <code>BASE_URL</code></small><br><br>
                <strong>API Endpoint:</strong><br>
                <div style="background: #f5f5f5; padding: 10px; border-radius: 5px; margin: 10px 0;">
                    <code style="font-size: 14px;"><?= (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'] . dirname($_SERVER['PHP_SELF']) . '/api.php' ?></code>
                </div>
                <small>Test API: <a href="<?= (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'] . dirname($_SERVER['PHP_SELF']) . '/api.php?action=classes' ?>" target="_blank">Click here</a></small><br><br>
                <strong>⚠️ Important:</strong> Delete this install.php file for security!
            </div>
        <?php elseif ($step == 1): ?>
            <div class="info">
                <strong>Welcome to EduMaster Installation!</strong><br><br>
                This installer will:
                <ul style="margin-left: 20px; margin-top: 10px;">
                    <li>Create database and tables</li>
                    <li>Set up configuration file</li>
                    <li>Create admin account</li>
                </ul>
            </div>
            <form method="get">
                <input type="hidden" name="step" value="2">
                <button type="submit" class="btn">Start Installation →</button>
            </form>
        <?php else: ?>
            <form method="post">
                <div class="form-group">
                    <label>Database Host</label>
                    <input type="text" name="db_host" value="localhost" required>
                </div>
                <div class="form-group">
                    <label>Database Name</label>
                    <input type="text" name="db_name" value="edumaster_db" required>
                </div>
                <div class="form-group">
                    <label>Database User</label>
                    <input type="text" name="db_user" value="root" required>
                </div>
                <div class="form-group">
                    <label>Database Password</label>
                    <input type="password" name="db_pass" value="">
                </div>
                
                <hr style="margin: 30px 0; border: 1px solid #e0e0e0;">
                <h3 style="color: #667eea; margin-bottom: 20px;">Super Admin Details</h3>
                
                <div class="form-group">
                    <label>Admin ID</label>
                    <input type="number" name="admin_id" value="1" required>
                    <small style="color: #666;">Unique ID for super admin</small>
                </div>
                <div class="form-group">
                    <label>Admin Name</label>
                    <input type="text" name="admin_name" value="Super Admin" required>
                </div>
                <div class="form-group">
                    <label>Admin Email (Username)</label>
                    <input type="email" name="admin_email" value="admin@edumaster.com" required>
                </div>
                <div class="form-group">
                    <label>Admin Password</label>
                    <input type="password" name="admin_password" value="admin123" required>
                </div>
                <input type="hidden" name="install" value="1">
                <button type="submit" class="btn">Install Now</button>
            </form>
        <?php endif; ?>
    </div>
</body>
</html>
